<?php 


class InvoicePDF extends FPDF {
    // Define variables
    private $company_info = array();
    private $customer_info = array();
    private $invoice_info = array();
    private $items = array();
    private $totals = array();
    private $header_image = '';
    private $footer_image = '';

    function __construct($orientation = 'P', $unit = 'mm', $size = 'A4') {
        parent::__construct($orientation, $unit, $size);
        $this->SetAutoPageBreak(true, 40); // Set bottom margin for footer
    }

    // Set header image
    function setHeaderImage($image) {
        $this->header_image = $image;
    }

    // Set footer image
    function setFooterImage($image) {
        $this->footer_image = $image;
    }

    // Set company info
    function setCompanyInfo($info) {
        $this->company_info = $info;
    }

    // Set customer info
    function setCustomerInfo($info) {
        $this->customer_info = $info;
    }

    // Set invoice info
    function setInvoiceInfo($info) {
        $this->invoice_info = $info;
    }

    // Add item to invoice
    function addItem($item) {
        $this->items[] = $item;
    }

    // Set totals
    function setTotals($totals) {
        $this->totals = $totals;
    }

    // Page header
    function Header() {
       if($this->header_image) {
            // Get image dimensions to maintain aspect ratio
            list($width, $height) = getimagesize($this->header_image);
            $ratio = $width / $height;
            
            // Set logo dimensions - smaller for top right positioning
            $new_width = 50; // Adjust this value as needed
            $new_height = $new_width / $ratio;
            
            // Position in top right corner with margins
            $this->Image($this->header_image, $this->GetPageWidth() - $new_width - 10, 10, $new_width, $new_height);
            
            // No line break after logo since it's in the corner
        } else {
            $this->Ln(10); // Space at top if no image
        }
    }

    // Page footer
    function Footer() {
         if($this->footer_image) {
            $this->SetY(-50); // Position at 30mm from bottom
            
            // Get image dimensions to maintain aspect ratio
            list($width, $height) = getimagesize($this->footer_image);
            $ratio = $width / $height;
            
            // Set dimensions to full page width
            $new_width = $this->GetPageWidth(); // Full width minus margins
            $new_height = $new_width / $ratio;
            
            // Add footer image centered at full width
            $this->Image($this->footer_image, 0, $this->GetY(), $new_width, $new_height);
        }
    }

    // Build the invoice
    function buildInvoice() {
        $this->AliasNbPages();
        $this->AddPage();
        
        // Company Information
        $this->SetFont('Arial', 'B', 12);
        //$this->Cell(0, 10, $this->company_info['name'], 0, 1);
        $this->SetFont('Arial', '', 9);
        //$this->Cell(0, 5, $this->company_info['address'], 0, 1);
        //$this->Cell(0, 5, $this->company_info['city'] . ', ' . $this->company_info['state'] . ' ' . $this->company_info['zip'], 0, 1);
        $this->Cell(0, 5, 'DeeSAT', 0, 1);
        $this->Cell(0, 5, 'Email: ' . $this->company_info['email'], 0, 1);
        $this->Ln(55);
        
        // Invoice Title
        $this->SetFont('Arial', 'B', 16);
        $this->Cell(0, 10, 'INVOICE', 0, 1, 'C');
        $this->Ln(5);
        
        // Invoice and Customer Info in two columns
        $this->SetFont('Arial', 'B', 10);
        
        // Left Column - Invoice Info
        $this->Cell(95, 7, 'Invoice Information:', 0, 0);
        // Right Column - Customer Info
        $this->Cell(95, 7, 'Bill To:', 0, 1);
        
        $this->SetFont('Arial', '', 9);
        
        // Invoice Info
        $this->Cell(95, 5, 'Tracking #: ' . $this->invoice_info['tracking_number'], 0, 0);
        // Customer Info
        $this->Cell(95, 5, $this->customer_info['name'], 0, 1);
        
        $this->Cell(95, 5, 'Order ID: ' . $this->invoice_info['order_id'], 0, 0);
        $this->Cell(95, 5, $this->customer_info['number'], 0, 1);
        
        $this->Cell(95, 5, 'Bank Reference No: ' . $this->invoice_info['bank_reference_no'], 0, 0);
        $this->Cell(95, 5, $this->customer_info['email'], 0, 1);
        
        $this->Cell(95, 5, 'Purchase Date: '.$this->invoice_info['date'], 0, 0);
        //$this->Cell(95, 5, $this->customer_info['email'], 0, 1);
        
        $this->Ln(10);
        
        // Items Table Header
        $this->SetFillColor(240, 240, 240);
        $this->SetFont('Arial', 'B', 10);
        $this->Cell(15, 8, 'Item #', 1, 0, 'C', true);
        $this->Cell(140, 8, 'Description', 1, 0, 'C', true);
        //$this->Cell(25, 8, 'Qty', 1, 0, 'C', true);
        //$this->Cell(30, 8, 'Unit Price', 1, 0, 'C', true);
        $this->Cell(35, 8, 'Amount', 1, 1, 'C', true);
        
        // Items
        $this->SetFont('Arial', '', 9);
        foreach($this->items as $item) {
            $this->Cell(15, 8, $item['id'], 1, 0, 'C');
            $this->Cell(140, 8, $item['description'], 1, 0, 'L');
            //$this->Cell(25, 8, $item['quantity'], 1, 0, 'C');
            //$this->Cell(30, 8, '$' . number_format($item['unit_price'], 2), 1, 0, 'R');
            $this->Cell(35, 8, $item['currency'] . number_format(1 * $item['unit_price'], 2), 1, 1, 'R');
        }
        
        // Totals
        // //$this->SetFont('Arial', 'B', 10);
        // $this->Cell(155, 8, 'Subtotal:', 1, 0, 'R');
        // $this->Cell(35, 8, '$' . number_format($this->totals['subtotal'], 2), 1, 1, 'R');
        
        // $this->Cell(155, 8, 'Tax (' . $this->totals['tax_rate'] . '%):', 1, 0, 'R');
        // $this->Cell(35, 8, '$' . number_format($this->totals['tax'], 2), 1, 1, 'R');
        
        $this->Cell(155, 8, 'Total:', 1, 0, 'R');
        $this->Cell(35, 8, $item['currency'] . number_format($this->totals['total'], 2), 1, 1, 'R');
        
        // Payment Instructions
        $this->Ln(10);
        $this->SetFont('Arial', 'B', 10);
        //$this->Cell(0, 8, 'Payment Instructions:', 0, 1);
        $this->SetFont('Arial', '', 9);
        $this->MultiCell(0, 5, '*This is a computer generated invoice and does not require a signature.');
        
        // Notes
        if(!empty($this->invoice_info['notes'])) {
            $this->Ln(25);
            $this->SetFont('Arial', 'B', 10);
            $this->Cell(0, 8, 'Notes:', 0, 1);
            $this->SetFont('Arial', 'B', 8);
            $this->Cell(0, 5,'DeeSAT - Exclusive SAT Prep for 1530+ Score Aspirants: Empowering ambitious students to achieve Top 1% SAT scores!');
            $this->Ln(4);
            $this->Cell(0, 5,'Have any queries? Email: contact@deesat.com. Check out our SAT prep packages: www.deesat.com. CIN: U80903MP2012PTC029454');
            
        }
    }
}


// Example usage
function generateInvoice($invoice_data) {
    $pdf = new InvoicePDF();
    
    // Set header and footer images
    if(isset($invoice_data['header_image']) && file_exists($invoice_data['header_image'])) {
        $pdf->setHeaderImage($invoice_data['header_image']);
    }
    
    if(isset($invoice_data['footer_image']) && file_exists($invoice_data['footer_image'])) {
        $pdf->setFooterImage($invoice_data['footer_image']);
    }
    
    // Set company info
    $pdf->setCompanyInfo($invoice_data['company_info']);
    
    // Set customer info
    $pdf->setCustomerInfo($invoice_data['customer_info']);
    
    // Set invoice info
    $pdf->setInvoiceInfo($invoice_data['invoice_info']);
    
    // Add items
    foreach($invoice_data['items'] as $item) {
        $pdf->addItem($item);
    }
    
    // Set totals
    $pdf->setTotals($invoice_data['totals']);
    
    // Build the invoice
    $pdf->buildInvoice();
    
    // Output the PDF
    if(isset($invoice_data['output_file'])) {
        $pdf->Output('F', $invoice_data['output_file']);
        return true;
    } else {
        $pdf->Output('F', 'invoices/invoice_' . $invoice_data['invoice_info']['number'] . '.pdf');
        return true;
    }
}

?>